/*-*- comment-column: 40 -*-*/
/* ipgen -- IP test packet description generator, main module. 
 *
 * Author: Juergen Nickelsen <nickelsen@condat.de>
 * Copyright (C) Condat AG 2000
 * $Id: main.c,v 1.6 2000/07/13 20:24:56 ni Exp $
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sysexits.h>
#include <stdio.h>
#include <string.h>
#include <stddef.h>
#include <assert.h>
#include <errno.h>
#include <stdlib.h>
#include <ctype.h>

/* Definition of packet data structure.  */
#include "ipgen.h"

/* Macro to define a header field struct instance. */
#define HFIELD(name) { #name, offsetof(struct IP_PKT, name) }

/* Array with header field structs, containing the name of the field
 * and the offset of the field in the IP_PKT struct. All these fields
 * can be handled uniformly -- there is one integer value that can be
 * assigned to the corresponding struct field. */
struct HEADERFIELD {
    char *name ;
    off_t offset ;
} hfield[] = {
    HFIELD(real_length),
    HFIELD(ip_v),
    HFIELD(ip_hl),
    HFIELD(ip_tos),
    HFIELD(ip_len),
    HFIELD(ip_id),
    HFIELD(ip_flags),
    HFIELD(ip_off),
    HFIELD(ip_ttl),
    HFIELD(ip_p),
    HFIELD(ip_sum),
    HFIELD(udp_sport),
    HFIELD(udp_dport),
    HFIELD(udp_ulen),
    HFIELD(udp_sum),
    HFIELD(icmp_type),
    HFIELD(icmp_code),
    HFIELD(icmp_sum),
    HFIELD(icmp_unused),
    HFIELD(icmp_id),
    HFIELD(icmp_seq),
    { NULL, 0 }
} ;


int output_width = DEFAULT_OUTPUT_WIDTH ; /* Number of bytes per output line. */


/** Safe strdup() function.
 * @param s   string to duplicate
 * @return    duplicated string
 */
char *ck_strdup(char *s)
{
    char *new = strdup(s) ;

    if (new == NULL) {
	fprintf(stderr, "ipgen: error: malloc failed in strdup()\n") ;
	exit(EX_TEMPFAIL) ;
    }
    return new ;
}


/** Safe strndup() function.
 * @param s   string to duplicate
 * @param n   number of characters to copy
 * @return    duplicated string
 */
char *ck_strndup(char *s, int n)
{
    char *new = malloc(n + 1) ;

    if (new == NULL) {
	fprintf(stderr, "ipgen: error: malloc failed in strndup()\n") ;
	exit(EX_TEMPFAIL) ;
    }
    memcpy(new, s, n) ;
    new[n] = '\0' ;
    return new ;
}



/** Skip non-whitespace characters in p. This includes a comma,
 * since we want to be able to use commas in the data area.
 * @param p  pointer to text area
 * @return   pointer to first non-whitespace character after p
 */
char *skip_whitespace(char *p)
{
    while (isspace(*p) || *p == ',') {
	p++ ;
    }
    return p ;
}


/** Return non-zero if character may be part of a keyword.
 * @param c   character to check
 * @return    non-zero iff c might be part of an identifier
 */
int is_identchar(char c)
{
    return isalnum(c) || c == '_' ;
}


/** Skip over all characters that may belong to an identifier (i. e. keyword)
 * @param p    pointer to text area
 * @return     pointer to first non-identchar character
 */
char *skip_identchars(char *p)
{
    while (is_identchar(*p)) {
	p++ ;
    }
    return p ;
}


/** Process a "data" line.
 * @param pkt_ptr  pointer to packet
 * @param fname    input file name (for error messages)
 * @param linenum  line number (for error messages)
 * @param line     input line (without data keyword)
 */
void process_data(struct IP_PKT *pkt_ptr, char *fname, int linenum, char *line)
{
    char *new ;				/* Pointer updated by strtol(). */
    unsigned val ;			/* Value read by strtol(). */

    for (;;) {
	/* Skip over commas and whitespace. */
	while (*line != '\0' && strchr(TOKSEP, *line)) {
	    line++ ;
	}
	val = strtol(line, &new, 16) ;
	if (new == line) {		/* strtol() has foudn no data. */
	    break ;
	} else {
	    line = new ;		/* Begin anew where strtol() stopped. */
	}

	if (val > 0xff) {
	    fprintf(stderr, "%s:%d: warning: data value > 0xff (%x)\n",
		    fname, linenum, val) ;
	}
	pkt_ptr->data[pkt_ptr->data_len++] = val ;
    }
}


/** Parse an input file and output the packets (via output_packet())
 * @param in      input stream
 * @param fname   name of input file
 * @return        number of errors
 */
int process_file(FILE *in, char *fname)
{
    int errors = 0 ;			/* Number of errors so far. */
    enum { INIT, DEFAULT, PACKET } state = INIT ;
    char line[LLENGTH+1] ;		/* Line buffer. */
    int linenum = 0 ;			/* Line number. */
    char *keyword ;			/* Keyword at start of line. */
    char *ptr ;				/* Pointer into line. */
    struct IP_PKT ip_pkt ;		/* IP packet to build. */
    struct IP_PKT def_pkt ;		/* Packet with default values. */
    struct IP_PKT *pkt_ptr = 0 ;	/* Pointer to current packet. */

    /* Initialize default packet. */
    memset(&def_pkt, '\0', sizeof(def_pkt)) ;

    while (fgets(line, LLENGTH, in)) {
	line[LLENGTH] = '\0' ;		/* Null-terminate line. */
	linenum++ ;

	/* If the newline char could not be read, the line is too
	 * long. Print an error and ignore the rest of the line. */
	if (line[strlen(line) - 1] != '\n') {
	    int c ;			/* A character read */
	    
	    fprintf(stderr, "%s:%d: warning: line longer than %d chars,"
		    " rest ignored\n", fname, linenum, LLENGTH) ;
	    errors++ ;
	    /* Read and ignore until newline. */
	    do {
		c = getc(in) ;
	    } while (c != EOF && c != '\n') ;
	} else {
	    /* Null out newline. */
	    line[strlen(line) - 1] = '\0' ;
	}

	/* Skip leading whitespace and check for comment or empty lines. */
	ptr = skip_whitespace(line) ;
	if (*ptr == '#' || *ptr == '\0') {
	    continue ;
	}

	/* Get keyword (needed in any state) and process line. */
	keyword = ptr ;
	ptr = skip_identchars(keyword) ;
	keyword = ck_strndup(keyword, ptr - keyword) ;

	switch (state) {
	  case INIT:
	    /* On top level. */
	    if (!strcmp(keyword, "default")) {
		state = DEFAULT ;
		pkt_ptr = &def_pkt ;
		/* Initialize default packet. */
		memset(&def_pkt, '\0', sizeof(def_pkt)) ;
		pkt_ptr->name = ck_strdup(skip_whitespace(ptr)) ;
	    } else if (!strcmp(keyword, "packet")) {
		state = PACKET ;
		pkt_ptr = &ip_pkt ;
		/* Initialize packet from default packet. */
		memcpy(&ip_pkt, &def_pkt, sizeof(ip_pkt)) ;
		pkt_ptr->real_length = -1 ; /* I. e. uninitialized. */
		pkt_ptr->name = ck_strdup(skip_whitespace(ptr)) ;
		if (strlen(pkt_ptr->name) == 0 && state == PACKET) {
		    fprintf(stderr,
			    "%s:%d: error: no packet name\n",
			    fname, linenum) ;
		    errors++ ;
		}
	    } else {
		fprintf(stderr,
			"%s:%d: error: invalid keyword %s outside of packet\n",
			fname, linenum, keyword) ;
		errors++ ;
	    }
	    break ;
	    
	  case PACKET:
	  case DEFAULT:
	    /* In packet or default packet. First check for keywords
	     * that have to be handled specially (end, data, ip_src,
	     * ip_dst, comment), then search in list of standard
	     * header fields. */
	    
	    if (!strcmp(keyword, "end")) {
		if (state == PACKET && errors == 0) {
		    output_packet(pkt_ptr) ;
		}
		state = INIT ;
	    } else if (!strcmp(keyword, "data")) {
		process_data(pkt_ptr, fname, linenum, skip_whitespace(ptr)) ;
	    } else if (!strcmp(keyword, "comment")) {
		if (pkt_ptr->comment) {
		    fprintf(stderr, "%s:%d: warning: duplicate comment"
			    " for packet %s\n",
			    fname, linenum, pkt_ptr->name) ;
		}
		pkt_ptr->comment = ck_strdup(skip_whitespace(ptr)) ;
	    } else if (!strcmp(keyword, "ip_src")) {
		ptr = skip_whitespace(ptr) ;
		if (!inet_aton(ptr, &pkt_ptr->ip_src)) {
		    fprintf(stderr, "%s:%d: error: invalid IP src addr %s\n",
			    fname, linenum, ptr) ;
		    errors++ ;
		}
	    } else if (!strcmp(keyword, "ip_dst")) {
		ptr = skip_whitespace(ptr) ;
		if (!inet_aton(ptr, &pkt_ptr->ip_dst)) {
		    fprintf(stderr, "%s:%d: error: invalid IP dst addr %s\n",
			    fname, linenum, ptr) ;
		    errors++ ;
		}
	    } else {			/* Standard fields. */
		int i ;			/* Index of keyword. */
		char *dummy ;

		/* Search keyword in list of known fields; if found,
		 * assign to corresponding field of packet struct. */
		for (i = 0; hfield[i].name != NULL; i++) {
		    if (!strcmp(keyword, hfield[i].name)) {
			unsigned value = strtol(ptr, &dummy, 0) ;

			if (dummy == ptr) {
			    fprintf(stderr, "%s:%d: error: missing %s value\n",
				    fname, linenum, keyword) ;
			    errors++ ;
			}
			*(unsigned *)((char *) pkt_ptr + hfield[i].offset) =
			    value ;
			if (!strcmp(keyword, "ip_sum")) {
			    pkt_ptr->has_ip_sum = 1 ;
			} else if (!strcmp(keyword, "udp_sum")) {
			    pkt_ptr->has_udp_sum = 1 ;
			} else if (!strcmp(keyword, "icmp_sum")) {
			    pkt_ptr->has_icmp_sum = 1 ;
			} else if (!strcmp(keyword, "ip_len")) {
			    pkt_ptr->has_ip_len = 1 ;
			} else if (!strcmp(keyword, "udp_ulen")) {
			    pkt_ptr->has_udp_ulen = 1 ;
			} else if (!strcmp(keyword, "icmp_id")) {
			    pkt_ptr->has_icmp_id = 1 ;
			}
			break ;
		    }
		}
		/* Complain if not found. */
		if (hfield[i].name == NULL) {
		    fprintf(stderr,
			    "%s:%d: error: unknown keyword %s in %spacket\n",
			    fname, linenum, keyword,
			    state == PACKET ? "" : "default ") ;
		    errors++ ;
		}
	    }
	    break ;
	  default:
	    fprintf(stderr, "%s:%d: error: impossible state %d\n",
		    fname, linenum, state) ;
	    exit(EX_SOFTWARE) ;
	    break ;
	}
	free(keyword) ;
    }
    if (ferror(in)) {
	fprintf(stderr, "ipgen: file error %s in %s, line %d\n",
		strerror(errno), fname, linenum) ;
	errors++ ;
    } else if (state == PACKET || state == DEFAULT) {
	fprintf(stderr, "%s:%d: error: EOF while reading %spacket\n",
		fname, linenum, state == DEFAULT ? "default " : "") ;
	errors++ ;
    }	
	    
    return errors ;
}


/** open input files and hand them to the parser.
 * @param argc
 * @param argv
 * @return        error status
 */
int main(int argc, char *argv[])
{
    int arg ;				/* Argument currently processed. */
    FILE *in ;				/* Input file. */
    char *fname ;			/* Name of current input file. */
    int errors = 0 ;			/* Number of errors encountered. */

    assert(sizeof(unsigned) == 4) ;
    srandom(1) ;

    /* Loop over arguments. Complain and exit if a file cannot be
     * opened. */
    for (arg = 1; arg < argc; arg++) {
	if (!strcmp(argv[arg], "-")) {
	    in = stdin ;
	    fname = "<stdin>" ;
	} else if ((in = fopen(argv[arg], "r")) == NULL) {
	    fprintf(stderr, "ipgen: can't read file %s (%s)\n",
		    argv[arg], strerror(errno)) ;
	    exit(EX_NOINPUT) ;
	}
	fname = argv[arg] ;
	printf("\n\n/* IPGEN: processing input file \"%s\" */\n", fname) ;
	errors += process_file(in, fname) ;
    }
    if (arg == 1) {
	fprintf(stderr, "ipgen: no input files\n") ;
    }
    
    return errors ? EX_DATAERR : 0 ;
}

/* EOF */
